<?php
/**
 * Plugin Name: WC Simple Order Tracking
 * Plugin URI:  https://www.techslideitsolutions.com/plugins/wc-simple-order-tracking
 * Description: A professional order tracking UI for WooCommerce. Track by Order ID with courier and item details.
 * Version:     1.4.1
 * Author:      TechSlide IT Solutions
 * Author URI:  https://www.techslideitsolutions.com
 * License:     GPL-2.0+
 * Text Domain: wc-simple-order-tracking
 * Requires at least: 5.8
 * Tested up to: 6.7
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Main Plugin Class
 */
class WCSimpleOrderTracking {

    private $option_name  = 'wc_ot_license_key';
    private $email_option = 'wc_ot_user_email';
    private $name_option  = 'wc_ot_user_name';
    private $version      = '1.4.1';
    private $slug         = 'wc-simple-order-tracking';
    private $api_endpoint = 'https://www.techslideitsolutions.com/wp-json/techslide/v1/validate-license';
    private $reg_endpoint = 'https://www.techslideitsolutions.com/wp-json/techslide/v1/register-license';
    private $fixed_plugin_name = 'WC Simple Order Tracking';

    public function __construct() {
        add_shortcode( 'woocommerce_order_tracker', array( $this, 'order_tracker_shortcode' ) );
        
        // Admin & Licensing
        add_action( 'admin_menu', array( $this, 'add_settings_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'init', array( $this, 'listen_for_cache_clear' ) );
    }

    /**
     * Clear validation cache remotely via Server Ping
     */
    public function listen_for_cache_clear() {
        if ( isset($_GET['ts_ot_clear_cache']) && $_GET['ts_ot_clear_cache'] == '1' ) {
            $this->force_clear_validation_cache();
        }
    }

    private function force_clear_validation_cache() {
        $key   = get_option( $this->option_name );
        $email = get_option( $this->email_option );
        if ( !empty($key) && !empty($email) ) {
            delete_transient( 'wc_ot_val_' . md5( $key . $email ) );
        }
    }

    /**
     * Validate license with Server API
     */
    private function validate_license_remotely( $key, $email ) {
        if ( empty( $key ) || empty( $email ) ) return false;

        $cache_key = 'wc_ot_val_' . md5( $key . $email );
        $cached_status = get_transient( $cache_key );
        if ( false !== $cached_status ) return ( $cached_status === 'valid' );

        $response = wp_remote_post( $this->api_endpoint, array(
            'body' => array(
                'license_key' => $key,
                'user_email'  => $email,
                'site_url'    => home_url(),
                'admin_email' => get_option('admin_email')
            ),
            'timeout' => 15
        ));

        if ( is_wp_error( $response ) ) return false;

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        $is_valid = isset( $body['status'] ) && $body['status'] === 'active';
        
        if ($is_valid && isset($body['user_name'])) {
            update_option($this->name_option, sanitize_text_field($body['user_name']));
        }

        set_transient( $cache_key, $is_valid ? 'valid' : 'invalid', 10 * MINUTE_IN_SECONDS );
        return $is_valid;
    }

    public function add_settings_menu() {
        add_options_page( 
            __( 'Order Tracker License', 'wc-simple-order-tracking' ), 
            __( 'Order Tracker License', 'wc-simple-order-tracking' ), 
            'manage_options', 
            'wc-order-tracker-license', 
            array( $this, 'settings_page_html' ) 
        );
    }

    /**
     * Register settings with mandatory sanitization callbacks
     */
    public function register_settings() {
        register_setting( 'wc_ot_group', $this->option_name, array(
            'sanitize_callback' => 'sanitize_text_field',
        ) );
        register_setting( 'wc_ot_group', $this->email_option, array(
            'sanitize_callback' => 'sanitize_email',
        ) );
        register_setting( 'wc_ot_group', $this->name_option, array(
            'sanitize_callback' => 'sanitize_text_field',
        ) );
    }

    private function get_country_codes() {
        return array(
            "93" => "Afghanistan (+93)", "355" => "Albania (+355)", "213" => "Algeria (+213)", "376" => "Andorra (+376)", "244" => "Angola (+244)", "1264" => "Anguilla (+1264)", "1268" => "Antigua & Barbuda (+1268)", "54" => "Argentina (+54)", "374" => "Armenia (+374)", "297" => "Aruba (+297)", "61" => "Australia (+61)", "43" => "Austria (+43)", "994" => "Azerbaijan (+994)", "1242" => "Bahamas (+1242)", "973" => "Bahrain (+973)", "880" => "Bangladesh (+880)", "1246" => "Barbados (+1246)", "375" => "Belarus (+375)", "32" => "Belgium (+32)", "501" => "Belize (+501)", "229" => "Benin (+229)", "1441" => "Bermuda (+1441)", "975" => "Bhutan (+975)", "591" => "Bolivia (+591)", "387" => "Bosnia Herzegovina (+387)", "267" => "Botswana (+267)", "55" => "Brazil (+55)", "673" => "Brunei (+673)", "359" => "Bulgaria (+359)", "226" => "Burkina Faso (+226)", "257" => "Burundi (+257)", "855" => "Cambodia (+855)", "237" => "Cameroon (+237)", "1" => "Canada (+1)", "238" => "Cape Verde Islands (+238)", "1345" => "Cayman Islands (+1345)", "236" => "Central African Republic (+236)", "56" => "Chile (+56)", "86" => "China (+86)", "57" => "Colombia (+57)", "269" => "Comoros (+269)", "242" => "Congo (+242)", "682" => "Cook Islands (+682)", "506" => "Costa Rica (+506)", "385" => "Croatia (+385)", "53" => "Cuba (+53)", "90392" => "Cyprus North (+90392)", "357" => "Cyprus South (+357)", "420" => "Czech Republic (+420)", "45" => "Denmark (+45)", "253" => "Djibouti (+253)", "1809" => "Dominica (+1809)", "1809" => "Dominican Republic (+1809)", "593" => "Ecuador (+593)", "20" => "Egypt (+20)", "503" => "El Salvador (+503)", "240" => "Equatorial Guinea (+240)", "291" => "Eritrea (+291)", "372" => "Estonia (+372)", "251" => "Ethiopia (+251)", "500" => "Falkland Islands (+500)", "298" => "Faroe Islands (+298)", "679" => "Fiji (+679)", "358" => "Finland (+358)", "33" => "France (+33)", "594" => "French Guiana (+594)", "689" => "French Polynesia (+689)", "241" => "Gabon (+241)", "220" => "Gambia (+220)", "995" => "Georgia (+995)", "49" => "Germany (+49)", "233" => "Ghana (+233)", "350" => "Gibraltar (+350)", "30" => "Greece (+30)", "299" => "Greenland (+299)", "1473" => "Grenada (+1473)", "590" => "Guadeloupe (+590)", "671" => "Guam (+671)", "502" => "Guatemala (+502)", "224" => "Guinea (+224)", "245" => "Guinea - Bissau (+245)", "592" => "Guyana (+592)", "509" => "Haiti (+509)", "504" => "Honduras (+504)", "852" => "Hong Kong (+852)", "36" => "Hungary (+36)", "354" => "Iceland (+354)", "91" => "India (+91)", "62" => "Indonesia (+62)", "98" => "Iran (+98)", "964" => "Iraq (+964)", "353" => "Ireland (+353)", "972" => "Israel (+972)", "39" => "Italy (+39)", "1876" => "Jamaica (+1876)", "81" => "Japan (+81)", "962" => "Jordan (+962)", "7" => "Kazakhstan (+7)", "254" => "Kenya (+254)", "686" => "Kiribati (+686)", "850" => "Korea North (+850)", "82" => "Korea South (+82)", "965" => "Kuwait (+965)", "996" => "Kyrgyzstan (+996)", "856" => "Laos (+856)", "371" => "Latvia (+371)", "961" => "Lebanon (+961)", "266" => "Lesotho (+266)", "231" => "Liberia (+231)", "218" => "Libya (+218)", "417" => "Liechtenstein (+417)", "370" => "Lithuania (+370)", "352" => "Luxembourg (+352)", "853" => "Macao (+853)", "389" => "Macedonia (+389)", "261" => "Madagascar (+261)", "265" => "Malawi (+265)", "60" => "Malaysia (+60)", "960" => "Maldives (+960)", "223" => "Mali (+223)", "356" => "Malta (+356)", "692" => "Marshall Islands (+692)", "596" => "Martinique (+596)", "222" => "Mauritania (+222)", "269" => "Mayotte (+269)", "52" => "Mexico (+52)", "691" => "Micronesia (+691)", "373" => "Moldova (+373)", "377" => "Monaco (+377)", "976" => "Mongolia (+976)", "1664" => "Montserrat (+1664)", "212" => "Morocco (+212)", "258" => "Mozambique (+258)", "95" => "Myanmar (+95)", "264" => "Namibia (+264)", "674" => "Nauru (+674)", "977" => "Nepal (+977)", "31" => "Netherlands (+31)", "687" => "New Caledonia (+687)", "64" => "New Zealand (+64)", "505" => "Nicaragua (+505)", "227" => "Niger (+227)", "234" => "Nigeria (+234)", "683" => "Niue (+683)", "672" => "Norfolk Islands (+672)", "670" => "Northern Marianas (+670)", "47" => "Norway (+47)", "968" => "Oman (+968)", "92" => "Pakistan (+92)", "680" => "Palau (+680)", "507" => "Panama (+507)", "675" => "Papua New Guinea (+675)", "595" => "Paraguay (+595)", "51" => "Peru (+51)", "63" => "Philippines (+63)", "48" => "Poland (+48)", "351" => "Portugal (+351)", "1787" => "Puerto Rico (+1787)", "974" => "Qatar (+974)", "262" => "Reunion (+262)", "40" => "Romania (+40)", "7" => "Russia (+7)", "250" => "Rwanda (+250)", "378" => "San Marino (+378)", "239" => "Sao Tome & Principe (+239)", "966" => "Saudi Arabia (+966)", "221" => "Senegal (+221)", "381" => "Serbia (+381)", "248" => "Seychelles (+248)", "232" => "Sierra Leone (+232)", "65" => "Singapore (+65)", "421" => "Slovak Republic (+421)", "386" => "Slovenia (+386)", "677" => "Solomon Islands (+677)", "252" => "Somalia (+252)", "27" => "South Africa (+27)", "34" => "Spain (+34)", "94" => "Sri Lanka (+94)", "290" => "St. Helena (+290)", "1869" => "St. Kitts (+1869)", "1758" => "St. Lucia (+1758)", "249" => "Sudan (+249)", "597" => "Suriname (+597)", "268" => "Swaziland (+268)", "46" => "Sweden (+46)", "41" => "Switzerland (+41)", "963" => "Syria (+963)", "886" => "Taiwan (+886)", "7" => "Tajikstan (+7)", "66" => "Thailand (+66)", "228" => "Togo (+228)", "676" => "Tonga (+676)", "1868" => "Trinidad & Tobago (+1868)", "216" => "Tunisia (+216)", "90" => "Turkey (+90)", "993" => "Turkmenistan (+993)", "1649" => "Turks & Caicos Islands (+1649)", "688" => "Tuvalu (+688)", "256" => "Uganda (+256)", "380" => "Ukraine (+380)", "971" => "United Arab Emirates (+971)", "44" => "United Kingdom (+44)", "1" => "USA (+1)", "598" => "Uruguay (+598)", "7" => "Uzbekistan (+7)", "678" => "Vanuatu (+678)", "379" => "Vatican City (+379)", "58" => "Venezuela (+58)", "84" => "Vietnam (+84)", "1284" => "Virgin Islands - British (+1284)", "1340" => "Virgin Islands - US (+1340)", "681" => "Wallis & Futuna (+681)", "969" => "Yemen (North)(+969)", "967" => "Yemen (South)(+967)", "260" => "Zambia (+260)", "263" => "Zimbabwe (+263)"
        );
    }

    public function settings_page_html() {
        $active_tab = isset( $_GET['tab'] ) ? $_GET['tab'] : 'license';
        
        if ( isset($_GET['refresh_license']) ) {
            $this->force_clear_validation_cache();
            wp_redirect(admin_url('options-general.php?page=wc-order-tracker-license&refreshed=1'));
            exit;
        }

        $current_key   = get_option( $this->option_name );
        $current_email = get_option( $this->email_option );
        $current_name  = get_option( $this->name_option );
        $status        = $this->validate_license_remotely( $current_key, $current_email );
        
        $reg_message = '';
        if ( isset($_POST['ts_reg_nonce']) && wp_verify_nonce($_POST['ts_reg_nonce'], 'ts_reg_action') ) {
            
            $country_code = sanitize_text_field($_POST['reg_country_code']);
            $mobile_raw   = sanitize_text_field($_POST['reg_whatsapp_raw']);
            $full_mobile  = '+' . $country_code . ltrim($mobile_raw, '0');

            $response = wp_remote_post($this->reg_endpoint, array(
                'body' => array(
                    'email'       => sanitize_email($_POST['reg_email']),
                    'name'        => sanitize_text_field($_POST['reg_name']),
                    'whatsapp'    => $full_mobile,
                    'plugin_name' => $this->fixed_plugin_name,
                    'site_url'    => home_url()
                ),
                'timeout' => 20
            ));

            if (is_wp_error($response)) {
                $reg_message = '<div class="notice notice-error"><p>' . esc_html($response->get_error_message()) . '</p></div>';
            } else {
                $data = json_decode(wp_remote_retrieve_body($response), true);
                if (isset($data['success']) && $data['success']) {
                    update_option($this->email_option, sanitize_email($_POST['reg_email']));
                    update_option($this->option_name, sanitize_text_field($data['license_key']));
                    $this->force_clear_validation_cache();
                    $reg_message = '<div class="notice notice-success"><p>' . esc_html__('Registration successful!', 'wc-simple-order-tracking') . '</p></div>';
                    $status = true;
                    $current_key = $data['license_key'];
                    $current_email = sanitize_email($_POST['reg_email']);
                } else {
                    $reg_message = '<div class="notice notice-error"><p>' . esc_html($data['message']) . '</p></div>';
                }
            }
        }
        ?>
        <div class="wrap">
            <h1><?php _e('WC Simple Order Tracking', 'wc-simple-order-tracking'); ?></h1>
            
            <?php if ( isset($_GET['refreshed']) ) echo '<div class="notice notice-info is-dismissible"><p>'.__('License status refreshed.', 'wc-simple-order-tracking').'</p></div>'; ?>

            <?php if ( $status ) : ?>
                <div class="notice notice-success" style="margin-top:20px; border-left-color: #1e7e34;">
                    <p><strong><?php _e('Plugin Activated!', 'wc-simple-order-tracking'); ?></strong> <?php _e('Use shortcode:', 'wc-simple-order-tracking'); ?> <code style="padding: 2px 8px; background: #fff; border: 1px solid #ddd;">[woocommerce_order_tracker]</code></p>
                </div>
            <?php endif; ?>

            <h2 class="nav-tab-wrapper">
                <a href="?page=wc-order-tracker-license&tab=license" class="nav-tab <?php echo $active_tab == 'license' ? 'nav-tab-active' : ''; ?>"><?php _e('License Status', 'wc-simple-order-tracking'); ?></a>
                <?php if ( ! $status ) : ?>
                    <a href="?page=wc-order-tracker-license&tab=register" class="nav-tab <?php echo $active_tab == 'register' ? 'nav-tab-active' : ''; ?>"><?php _e('Register New License', 'wc-simple-order-tracking'); ?></a>
                <?php endif; ?>
            </h2>

            <?php if ($active_tab == 'license') : ?>
                <div style="background: #fff; border: 1px solid #ccd0d4; padding: 20px; margin-top: 20px; border-radius: 4px;">
                    <table class="form-table">
                        <?php if ( $status ) : ?>
                            <tr><th><?php _e('Verified User', 'wc-simple-order-tracking'); ?></th><td><strong><?php echo esc_html($current_name); ?></strong></td></tr>
                            <tr><th><?php _e('Email', 'wc-simple-order-tracking'); ?></th><td><?php echo esc_html($current_email); ?></td></tr>
                            <tr><th><?php _e('License Key', 'wc-simple-order-tracking'); ?></th><td><code><?php echo esc_html($current_key); ?></code></td></tr>
                        <?php else : ?>
                            <form method="post" action="options.php">
                            <?php settings_fields( 'wc_ot_group' ); ?>
                            <tr><th><?php _e('Registered Email', 'wc-simple-order-tracking'); ?></th><td><input type="email" name="<?php echo esc_attr($this->email_option); ?>" value="<?php echo esc_attr($current_email); ?>" class="regular-text" required /></td></tr>
                            <tr><th><?php _e('License Key', 'wc-simple-order-tracking'); ?></th><td><input type="text" name="<?php echo esc_attr($this->option_name); ?>" value="<?php echo esc_attr($current_key); ?>" class="regular-text" required /></td></tr>
                            <tr><th></th><td><?php submit_button(__('Save & Validate', 'wc-simple-order-tracking'), 'primary', 'submit', false); ?></td></tr>
                            </form>
                        <?php endif; ?>
                        
                        <tr>
                            <th><?php _e('Status', 'wc-simple-order-tracking'); ?></th>
                            <td>
                                <span style="display:inline-block; color: <?php echo $status ? 'green':'red'; ?>; font-weight: bold; background: <?php echo $status ? '#e7f4e9':'#fbeaea'; ?>; padding: 5px 12px; border-radius: 4px; border: 1px solid <?php echo $status ? '#c3e6cb':'#f5c6cb'; ?>;">
                                    <?php echo $status ? __('✓ Verified & Active', 'wc-simple-order-tracking') : __('✗ Inactive / Restricted', 'wc-simple-order-tracking'); ?>
                                </span>
                            </td>
                        </tr>
                    </table>
                    
                    <div style="margin-top: 20px; padding-top: 15px; border-top: 1px solid #eee;">
                        <a href="<?php echo admin_url('options-general.php?page=wc-order-tracker-license&refresh_license=1'); ?>" class="button button-secondary"><?php _e('Refresh License Status', 'wc-simple-order-tracking'); ?></a>
                    </div>
                </div>
            <?php else : ?>
                <div style="margin-top:20px; max-width:600px; background: #fff; border: 1px solid #ccd0d4; padding: 20px; border-radius: 4px;">
                    <p><?php printf( __('Register below to receive your license for %s.', 'wc-simple-order-tracking'), '<strong>'.$this->fixed_plugin_name.'</strong>'); ?></p>
                    <?php echo $reg_message; ?>
                    <form method="post" action="">
                        <?php wp_nonce_field('ts_reg_action', 'ts_reg_nonce'); ?>
                        <table class="form-table">
                            <tr><th><?php _e('Full Name', 'wc-simple-order-tracking'); ?></th><td><input type="text" name="reg_name" class="regular-text" placeholder="John Doe" required /></td></tr>
                            <tr><th><?php _e('Email', 'wc-simple-order-tracking'); ?></th><td><input type="email" name="reg_email" class="regular-text" placeholder="john@example.com" required /></td></tr>
                            <tr>
                                <th><?php _e('WhatsApp Number', 'wc-simple-order-tracking'); ?></th>
                                <td>
                                    <div style="display: flex; gap: 5px; align-items: center;">
                                        <select name="reg_country_code" style="max-width: 150px;" required>
                                            <option value=""><?php _e('Select Code', 'wc-simple-order-tracking'); ?></option>
                                            <?php foreach ($this->get_country_codes() as $code => $label): ?>
                                                <option value="<?php echo $code; ?>" <?php selected($code, '91'); ?>><?php echo esc_html($label); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                        <input type="text" name="reg_whatsapp_raw" class="regular-text" placeholder="9876543210" style="flex: 1;" required />
                                    </div>
                                    <p class="description"><?php _e('Select country code and enter mobile number without zero.', 'wc-simple-order-tracking'); ?></p>
                                </td>
                            </tr>
                        </table>
                        <?php submit_button(__('Register & Get Key', 'wc-simple-order-tracking')); ?>
                    </form>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    public function order_tracker_shortcode() {
        $key   = get_option( $this->option_name );
        $email = get_option( $this->email_option );
        if ( ! $this->validate_license_remotely( $key, $email ) ) {
            return '<div class="wc-ot-error">'.__('License invalid or suspended. Please contact administrator.', 'wc-simple-order-tracking').'</div>';
        }

        if ( ! class_exists( 'WooCommerce' ) ) return '<p class="wc-ot-error">'.__('WooCommerce is not active.', 'wc-simple-order-tracking').'</p>';

        $order_id = isset($_POST['ot_order_id']) ? absint($_POST['ot_order_id']) : 0;
        $order = $order_id > 0 ? wc_get_order( $order_id ) : null;
        $error = ($order_id > 0 && !$order) ? __('Order not found.', 'wc-simple-order-tracking') : '';

        ob_start();
        ?>
        <div class="ts-ot-wrapper">
            <div class="ts-ot-card">
                <div class="ts-ot-header"><h2><?php _e('Track Your Order', 'wc-simple-order-tracking'); ?></h2></div>
                <form class="ts-ot-form" method="post">
                    <div class="ts-ot-input-group">
                        <input type="text" name="ot_order_id" value="<?php echo $order_id ? esc_attr($order_id) : ''; ?>" placeholder="<?php _e('Order ID', 'wc-simple-order-tracking'); ?>" required>
                        <button type="submit"><?php _e('Track', 'wc-simple-order-tracking'); ?></button>
                    </div>
                </form>

                <?php if ($error) : ?><div class="ts-ot-error-msg"><?php echo esc_html($error); ?></div><?php endif; ?>

                <?php if ($order) : 
                    $status = $order->get_status();
                    $ast = $order->get_meta('_wc_shipment_tracking_items');
                    $provider = ''; $tracking = ''; $link = '';

                    if (!empty($ast) && is_array($ast)) {
                        $item = $ast[0];
                        $provider = !empty($item['formatted_tracking_provider']) ? $item['formatted_tracking_provider'] : (!empty($item['tracking_provider']) ? $item['tracking_provider'] : '');
                        $tracking = !empty($item['tracking_number']) ? $item['tracking_number'] : '';
                        $link = !empty($item['tracking_link']) ? $item['tracking_link'] : '';
                    } else {
                        $provider = $order->get_meta('_tracking_provider') ?: $order->get_meta('courier_provider');
                        $tracking = $order->get_meta('_tracking_number') ?: $order->get_meta('tracking_id');
                    }
                ?>
                    <div class="ts-ot-results">
                        <div class="ts-ot-status status-<?php echo esc_attr($status); ?>">
                            <span><?php echo __('Status:', 'wc-simple-order-tracking') . ' ' . esc_html(wc_get_order_status_name($status)); ?></span>
                        </div>
                        <div class="ts-ot-grid">
                            <div class="ts-ot-item"><span class="ts-ot-label"><?php _e('Order #', 'wc-simple-order-tracking'); ?></span><span class="ts-ot-value">#<?php echo $order->get_id(); ?></span></div>
                            <div class="ts-ot-item"><span class="ts-ot-label"><?php _e('Date', 'wc-simple-order-tracking'); ?></span><span class="ts-ot-value"><?php echo wc_format_datetime($order->get_date_created()); ?></span></div>
                            <?php if($provider): ?><div class="ts-ot-item"><span class="ts-ot-label"><?php _e('Courier', 'wc-simple-order-tracking'); ?></span><span class="ts-ot-value ts-ot-accent"><?php echo esc_html($provider); ?></span></div><?php endif; ?>
                            <?php if($tracking): ?><div class="ts-ot-item"><span class="ts-ot-label"><?php _e('Tracking #', 'wc-simple-order-tracking'); ?></span><span class="ts-ot-value ts-ot-accent"><?php if($link): ?><a href="<?php echo esc_url($link); ?>" target="_blank"><?php echo esc_html($tracking); ?> ↗</a><?php else: echo esc_html($tracking); endif; ?></span></div><?php endif; ?>
                            <div class="ts-ot-item"><span class="ts-ot-label"><?php _e('Total', 'wc-simple-order-tracking'); ?></span><span class="ts-ot-value"><?php echo $order->get_formatted_order_total(); ?></span></div>
                            <div class="ts-ot-item"><span class="ts-ot-label"><?php _e('Payment', 'wc-simple-order-tracking'); ?></span><span class="ts-ot-value"><?php echo esc_html($order->get_payment_method_title()); ?></span></div>
                        </div>
                        <div class="ts-ot-items-sec">
                            <h4><?php _e('Items Ordered', 'wc-simple-order-tracking'); ?></h4>
                            <ul class="ts-ot-list">
                                <?php foreach ( $order->get_items() as $item ) : ?>
                                    <li><span><?php echo esc_html($item->get_name()); ?></span><strong>x<?php echo esc_html($item->get_quantity()); ?></strong></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <style>
            .ts-ot-wrapper { max-width: 550px; margin: 30px auto; font-family: -apple-system, system-ui, sans-serif; }
            .ts-ot-card { background: #fff; padding: 30px; border-radius: 12px; box-shadow: 0 10px 30px rgba(0,0,0,0.05); border: 1px solid #eee; }
            .ts-ot-header h2 { text-align: center; margin: 0 0 25px; color: #333; }
            .ts-ot-input-group { display: flex; gap: 10px; margin-bottom: 20px; }
            .ts-ot-input-group input { flex: 1; padding: 12px; border: 2px solid #eee; border-radius: 8px; font-size: 16px; }
            .ts-ot-input-group button { background: #007cba; color: #fff; border: none; padding: 0 25px; border-radius: 8px; font-weight: bold; cursor: pointer; }
            .ts-ot-results { margin-top: 25px; border-top: 1px dashed #eee; padding-top: 25px; }
            .ts-ot-status { padding: 15px; border-radius: 10px; text-align: center; font-weight: bold; margin-bottom: 25px; }
            .status-completed { background: #e6f7ed; color: #1e7e34; }
            .status-processing { background: #fff7e6; color: #d48806; }
            .ts-ot-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 25px; }
            .ts-ot-item { display: flex; flex-direction: column; }
            .ts-ot-label { font-size: 11px; color: #999; text-transform: uppercase; letter-spacing: 0.5px; }
            .ts-ot-value { font-weight: bold; color: #333; margin-top: 2px; }
            .ts-ot-accent { color: #5b21b6; }
            .ts-ot-items-sec h4 { border-bottom: 1px solid #eee; padding-bottom: 10px; margin-bottom: 15px; }
            .ts-ot-list { list-style: none; padding: 0; margin: 0; }
            .ts-ot-list li { display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #fafafa; font-size: 14px; }
            .ts-ot-error-msg { background: #fff1f0; color: #cf1322; padding: 15px; border-radius: 8px; text-align: center; margin-bottom: 20px; }
        </style>
        <?php
        return ob_get_clean();
    }
}
new WCSimpleOrderTracking();